<?php
// app/Http/Controllers/api/LabTestCatalogController.php
// Lab tests are products with is_storable=0; parent_id for panels/sub-tests

namespace App\Http\Controllers\api;

use App\Http\Controllers\Controller;
use App\Models\Stock\Product;
use App\Models\Stock\ProductCategory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class LabTestCatalogController extends Controller
{
    protected function labTestsQuery($companyId)
    {
        return Product::withoutGlobalScope(\App\Scopes\CompanyScope::class)
            ->where('company_id', $companyId)
            ->where('is_storable', 0);
    }

    /**
     * Display a listing of main categories (parent lab tests).
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $user = auth()->user();
            $companyId = $user->company_id;

            $query = $this->labTestsQuery($companyId)->whereNull('parent_id');

            if ($request->filled('search')) {
                $query->where('name', 'LIKE', '%' . $request->search . '%');
            }

            if ($request->has('is_active')) {
                $query->where('status', $request->boolean('is_active') ? 1 : 0);
            }

            if ($request->boolean('with_children')) {
                $query->with(['children' => fn ($q) => $q->orderBy('name')]);
            }

            $query->withCount('children');

            $tests = $query->orderBy('name')->get()->map(function ($p) {
                return $this->formatLabTest($p);
            });

            return response()->json([
                'success' => true,
                'rows' => $tests,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve lab tests: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get sub-tests by parent ID.
     */
    public function showByParent($parentId, Request $request): JsonResponse
    {
        try {
            $user = auth()->user();
            $companyId = $user->company_id;

            $parent = $this->labTestsQuery($companyId)->find($parentId);
            if (!$parent) {
                return response()->json(['success' => false, 'message' => 'Parent test not found'], 404);
            }

            $subTests = $this->labTestsQuery($companyId)
                ->where('parent_id', $parentId)
                ->orderBy('name')
                ->get()
                ->map(fn ($p) => $this->formatLabTest($p));

            return response()->json(['success' => true, 'rows' => $subTests]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve sub-tests: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Store a new lab test (product with is_storable=0).
     */
    public function store(Request $request): JsonResponse
    {
        DB::beginTransaction();

        try {
            $user = auth()->user();
            $companyId = $user->company_id;

            $validator = Validator::make($request->all(), [
                'name' => [
                    'required', 'string', 'max:50',
                    function ($attr, $v, $fail) use ($companyId) {
                        $q = Product::withoutGlobalScope(\App\Scopes\CompanyScope::class)
                            ->where('company_id', $companyId)
                            ->where('is_storable', 0)
                            ->where('name', $v);
                        if ($q->exists()) {
                            $fail('A test with this name already exists.');
                        }
                    },
                ],
                'category' => 'required|string|max:50',
                'price' => 'nullable|numeric|min:0|max:99999999.99',
                'normal_range' => 'nullable|string|max:50',
                'description' => 'nullable|string|max:255',
                'parent_id' => 'nullable|exists:products,id',
                'is_active' => 'boolean',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $data = $validator->validated();
            $labCategory = ProductCategory::where('company_id', $companyId)->where('name', 'lab')->first();

            if (!empty($data['parent_id'])) {
                $parent = $this->labTestsQuery($companyId)->find($data['parent_id']);
                if (!$parent) {
                    return response()->json(['status' => false, 'message' => 'Parent test not found'], 422);
                }
            }

            $product = Product::withoutGlobalScope(\App\Scopes\CompanyScope::class)->create([
                'company_id'    => $companyId,
                'name'          => $data['name'],
                'test_category'  => $data['category'] ?? null,
                'category_id'   => optional($labCategory)->id,
                'parent_id'     => $data['parent_id'] ?? null,
                'private_price' => (float) ($data['price'] ?? 0),
                'quantity'      => 0,
                'is_storable'   => 0,
                'normal_range'  => $data['normal_range'] ?? null,
                'description'   => $data['description'] ?? null,
                'status'        => ($data['is_active'] ?? true) ? 1 : 0,
                'created_by'    => $user->id,
            ]);

            DB::commit();

            return response()->json([
                'status' => true,
                'row' => $this->formatLabTest($product->load('parent')),
                'message' => 'Lab test created successfully',
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to create lab test: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update the specified lab test.
     */
    public function update(Request $request, $id): JsonResponse
    {
        DB::beginTransaction();

        try {
            $user = auth()->user();
            $companyId = $user->company_id;

            $product = $this->labTestsQuery($companyId)->find($id);
            if (!$product) {
                return response()->json(['status' => false, 'message' => 'Lab test not found'], 404);
            }

            $validator = Validator::make($request->all(), [
                'name' => [
                    'required', 'string', 'max:50',
                    function ($attr, $v, $fail) use ($companyId, $id) {
                        $q = Product::withoutGlobalScope(\App\Scopes\CompanyScope::class)
                            ->where('company_id', $companyId)
                            ->where('is_storable', 0)
                            ->where('name', $v)
                            ->where('id', '!=', $id);
                        if ($q->exists()) {
                            $fail('A test with this name already exists.');
                        }
                    },
                ],
                'category' => 'required|string|max:50',
                'price' => 'nullable|numeric|min:0|max:99999999.99',
                'normal_range' => 'nullable|string|max:50',
                'description' => 'nullable|string|max:255',
                'parent_id' => 'nullable|exists:products,id',
                'is_active' => 'boolean',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $data = $validator->validated();

            if (!empty($data['parent_id'])) {
                if ((int) $data['parent_id'] === (int) $product->id) {
                    return response()->json(['status' => false, 'message' => 'A test cannot be its own parent'], 422);
                }
                $parent = $this->labTestsQuery($companyId)->find($data['parent_id']);
                if (!$parent) {
                    return response()->json(['status' => false, 'message' => 'Parent test not found'], 422);
                }
                if ($this->isDescendant($data['parent_id'], $product->id, $companyId)) {
                    return response()->json(['status' => false, 'message' => 'Cannot move test under its own descendant'], 422);
                }
            }

            $product->update([
                'name'          => $data['name'],
                'test_category'  => $data['category'] ?? null,
                'private_price' => (float) ($data['price'] ?? 0),
                'normal_range'  => $data['normal_range'] ?? null,
                'description'   => $data['description'] ?? null,
                'parent_id'     => $data['parent_id'] ?? null,
                'status'        => ($data['is_active'] ?? true) ? 1 : 0,
            ]);

            DB::commit();

            return response()->json([
                'status' => true,
                'row' => $this->formatLabTest($product->fresh()->load('parent')),
                'message' => 'Lab test updated successfully',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to update lab test: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove the specified lab test.
     */
    public function destroy($id): JsonResponse
    {
        DB::beginTransaction();

        try {
            $user = auth()->user();
            $companyId = $user->company_id;

            $product = $this->labTestsQuery($companyId)->withCount('children')->find($id);
            if (!$product) {
                return response()->json(['status' => false, 'message' => 'Lab test not found'], 404);
            }

            if ($product->children_count > 0) {
                return response()->json([
                    'status' => false,
                    'message' => 'Cannot delete test that has sub-tests',
                ], 422);
            }

            $product->delete();

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Lab test deleted successfully',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete lab test: ' . $e->getMessage(),
            ], 500);
        }
    }

    protected function formatLabTest($product)
    {
        $children = $product->relationLoaded('children') ? $product->children->map(fn ($c) => $this->formatLabTest($c)) : [];
        return [
            'id' => $product->id,
            'name' => $product->name,
            'category' => $product->test_category ?? optional($product->category)->name ?? null,
            'price' => $product->private_price,
            'normal_range' => $product->normal_range,
            'description' => $product->description,
            'parent_id' => $product->parent_id,
            'is_active' => (int) $product->status === 1,
            'children' => $children,
            'children_count' => $product->children_count ?? $product->children->count() ?? 0,
            'children_total' => $product->children_count ?? $product->children->count() ?? 0,
        ];
    }

    private function isDescendant($parentId, $testId, $companyId): bool
    {
        $parent = $this->labTestsQuery($companyId)->find($parentId);
        while ($parent) {
            if ((int) $parent->parent_id === (int) $testId) {
                return true;
            }
            $parent = $parent->parent;
        }
        return false;
    }
}
