<?php

namespace App\Http\Controllers\Stock;

use Carbon\Carbon;
use App\Models\Stock\Sale;
use App\Models\Stock\Stock;
use Illuminate\Http\Request;
use App\Models\Stock\Expense;
use App\Models\Stock\Payment;
use App\Models\Stock\Product;
use App\Models\Stock\SaleItem;
use Illuminate\Support\Facades\DB;
use App\Models\Stock\PaymentMethod;
use App\Models\Stock\ProductChange;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Cache;

class POSController extends Controller
{
     /**
     * Public function get Dashboard
     * @param Request $request
     * @return JsonResponse
     */
    public function getDashboardData(Request $request)
    {
        $yearMonth = date('Y-m');
        $year = date('Y');
        if(!empty($from = $request->get('yearMonth'))){
            $yearMonth = date('Y-m', strtotime($from));
            $year = date('Y', strtotime($from));
        }

        $pendingChart = $paidChart = $labels = [];
        $days = getMonthDays(date('m', strtotime($yearMonth)), $year);
        for ($i = 1; $i <= count($days); $i++) {
            $labels[] = sprintf("%02d", $i);
            $pendingChart[] = Sale::where("committed_date", $days[$i - 1])->where('paid', 0)->sum('discounted_total');   
            $paidChart[]    = Sale::where("committed_date", $days[$i - 1])->where('paid', 1)->sum('discounted_total');   
        }

        $today = now(); // Gets the current date and time
        $firstDayOfPreviousMonth = $today->subMonth()->startOfMonth(); // Gets the first day of the previous month
        $lastDayOfPreviousMonth = $firstDayOfPreviousMonth->copy()->endOfMonth(); // Gets the last day of the previous month

        $PaidInPreviousMonth = Sale::whereBetween('committed_date', [$firstDayOfPreviousMonth, $lastDayOfPreviousMonth])->where('paid', 1)->sum('discounted_total');
        $TotalInPreviousMonth = Sale::whereBetween('committed_date', [$firstDayOfPreviousMonth, $lastDayOfPreviousMonth])->sum('discounted_total');
        $PendingInPreviousMonth = Sale::whereBetween('committed_date', [$firstDayOfPreviousMonth, $lastDayOfPreviousMonth])->where('paid', 0)->sum('discounted_total');
        $ExpensesInPreviousMonth = Expense::whereBetween('committed_date', [$firstDayOfPreviousMonth, $lastDayOfPreviousMonth])->sum('amount');
        
        $sales_count    = Sale::where('committed_date', 'LIKE', "%{$yearMonth}%")->count();
        $total_paid     = Sale::where('committed_date', 'LIKE', "%{$yearMonth}%")->where('paid', 1)->sum('discounted_total');
        $total_pending  = Sale::where('committed_date', 'LIKE', "%{$yearMonth}%")->where('paid', 0)->sum('discounted_total');
        $total_expenses = Expense::where('committed_date', 'LIKE', "%{$yearMonth}%")->sum('amount');
        $total_sales    = Sale::where('committed_date', 'LIKE', "%{$yearMonth}%")->sum('discounted_total');

        $PosCache = [
            'PaidInPreviousMonth' =>(int)$PaidInPreviousMonth,
            'TotalInPreviousMonth' =>(int)$TotalInPreviousMonth,
            'PendingInPreviousMonth' =>(int)$PendingInPreviousMonth,
            'ExpensesInPreviousMonth' =>(int)$ExpensesInPreviousMonth,
            'sales_count'       =>(int)$sales_count,
            'total_paid'        =>(int)$total_paid,
            'total_pending'     =>(int)$total_pending,
            'total_expenses'    =>(int)$total_expenses,
            'total_sales'       =>(int)$total_sales
        ];

        Cache::put('PosCache', $PosCache ,60);

        return response()->json([
            'PaidInPreviousMonth' => $PaidInPreviousMonth,
            'TotalInPreviousMonth' =>(int)$TotalInPreviousMonth,
            'PendingInPreviousMonth' =>(int)$PendingInPreviousMonth,
            'ExpensesInPreviousMonth' =>(int)$ExpensesInPreviousMonth,
            'sales_count'    => $sales_count,
            'total_sales'    => $total_sales,
            'total_expenses' => intval($total_expenses),
            'total_paid'     => $total_paid,
            'total_pending'  => $total_pending,
            'labels'         => $labels,
            'pending_chart'  => $pendingChart,
            'paid_chart'     => $paidChart
        ]);
    }

    /**
     * Filter dashboard cards
     * @param Request $request
     * @return JsonResponse
     */
    public function filterDashboardCards(Request $request)
    {
        $from = Carbon::parse($request->input('from'));
        $to = Carbon::parse($request->input('to'));
    
        if ($from->greaterThanOrEqualTo($to)) {
            return response()->json([
                'status' => 0,
                'message'=> 'Invalid date range: "From" date must be before "To" date.'
            ], 400);
        }

        $from = $from->format('Y-m-d');
        $to = $to->format('Y-m-d');

        return response()->json([
            'sales_count'    => Sale::where('committed_date', '>=', $from)->where('committed_date', '<=', $to)->count(),
            'total_sales'    => Sale::where('committed_date', '>=', $from)->where('committed_date', '<=', $to)->sum('discounted_total'),
            'total_expenses' => Expense::where('committed_date', '>=', $from)->where('committed_date', '<=', $to)->sum('amount'),
            'total_paid'     => Sale::where('committed_date', '>=', $from)->where('committed_date', '<=', $to)->where('paid', 1)->sum('discounted_total'),
            'total_pending'  => Sale::where('committed_date', '>=', $from)->where('committed_date', '<=', $to)->where('paid', 0)->sum('discounted_total'),
        ]);
    }

     /**
     * Get All sales Items 
     * 
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function salesItems(Request $request)
    {
        // Storable products (is_storable=1) + parent services (is_storable=0, parent_id=null)
        // Exclude child products (parent_id not null) - they're part of panels only
        $storables = Product::select('id', 'name', 'packaging', 'image', 'reference', 'private_price', 'quantity', 'category_id', 'unit_id', 'is_storable', 'parent_id')
            ->where('is_storable', 1)
            ->orderBy('products.name', 'ASC')
            ->with('unit', 'category', 'expiry', 'ProductChanges')
            ->get()
            ->map(function ($p) {
                return [
                    'id'             => $p->id,
                    'name'           => $p->name,
                    'packaging'      => $p->packaging,
                    'image'          => $p->image,
                    'reference'      => $p->reference,
                    'price'          => (float) $p->private_price,
                    'qty'            => (int) $p->quantity,
                    'category_id'    => $p->category_id,
                    'unit_id'        => $p->unit_id,
                    'is_storable'    => 1,
                    'unit'           => $p->unit,
                    'category'       => $p->category,
                    'expiry'         => $p->expiry,
                    'ProductChanges' => $p->ProductChanges,
                    'product_changes' => $p->ProductChanges->toArray(),
                ];
            });

        $parentServices = Product::select('id', 'name', 'packaging', 'image', 'reference', 'private_price', 'quantity', 'category_id', 'unit_id', 'is_storable', 'parent_id')
            ->where('is_storable', 0)
            ->whereNull('parent_id')
            ->with('children', 'unit', 'category')
            ->orderBy('products.name', 'ASC')
            ->get()
            ->map(function ($p) {
                $childrenTotal = $p->children->sum('private_price');
                $price = (float) ($p->private_price ?? 0) + (float) $childrenTotal;
                return [
                    'id'              => $p->id,
                    'name'            => $p->name,
                    'packaging'       => $p->packaging,
                    'image'           => $p->image,
                    'reference'       => $p->reference,
                    'price'           => $price,
                    'qty'             => 9999, // unlimited for services
                    'category_id'     => $p->category_id,
                    'unit_id'         => $p->unit_id,
                    'is_storable'     => 0,
                    'unit'            => $p->unit,
                    'category'        => $p->category,
                    'expiry'          => null,
                    'ProductChanges'  => collect([]),
                    'product_changes' => [],
                ];
            });

        $rows = $storables->concat($parentServices)->sortBy('name')->values();

        return response()->json([
            'status' => 1,
            'rows'   => $rows
        ]);
    }

     /**
     * List all Orders
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $result = Sale::selectRaw('id, reference, code, committed_date, discounted_total, amount_paid, amount_remain, client_id, paid, branch_id, create_user, comment, payment_date');
               
        $paymentTotals = DB::table('payments')
                        ->join('payment_methods', 'payments.payment_type', '=', 'payment_methods.id')
                        ->join('sales', 'payments.transaction_id', '=', 'sales.id')
                        ->select('payment_methods.name as payment_type', DB::raw('SUM(payments.amount_paid) as total'))
                        ->groupBy('payment_type');

        $from = date('Y-m-d');
        if(!empty($fromd = $request->get('from'))){
            $from = $fromd;
        }

        $to = $request->get('to');
        if (empty($to)){
            $result->where('committed_date', $from);
            $paymentTotals->where('payments.committed_date', $from);
        } else {
            $result->where('committed_date', '>=', $from)
                    ->where('committed_date', '<=', $to);

            $paymentTotals->where('payments.committed_date', '>=', $from)
                    ->where('payments.committed_date', '<=', $to);
        }

        if (!empty($branch = $request->get('branch_id'))) {
            $result->where('branch_id', $branch);
        }
        if (!empty($client = $request->get('client_id'))) {
            $result->where('client_id', $client);
        }
        if (!empty($type = $request->get('type'))) {
            $result->where('type', $type);
        }
        if ($request->has('status')) {
            $status = $request->get('status');
            if($status=="PAID"){
                $result->where('paid', 1);
            }elseif($status=="DEBTS"){
                $result->where('paid', 0);
            }else{
                $result->onlyTrashed();
            }
        }
        if (empty($isAdmin = $request->get('is_admin'))) {
            $result->where('create_user', auth()->id());
        }
        return response()->json([
            'status' => 1,
            'rows'   => $result->with('client', 'creator')
                               ->orderBy('id', 'DESC')
                               ->paginate(45),
            'payments_methods' => $paymentTotals->get()
        ]);
    }

    /**
     * Place a new Order
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
{
    // First validate all items before processing
    $items = json_decode($request->items);
    
    // Check stock for storable items only (services have no stock)
    foreach($items as $item) {
        $product = Product::find($item->id);
        
        if (!$product) {
            return response()->json([
                'status' => 0,
                'message' => 'Product not found with ID: ' . $item->id
            ]);
        }
        
        // Skip stock check for services (is_storable=0)
        if ($product->is_storable) {
            if ($item->quantity > $product->quantity) {
                return response()->json([
                    'status' => 0,
                    'message' => 'Requested quantity for ' . $product->name . ' exceeds available stock. Available: ' . $product->quantity
                ]);
            }
        }
        
        // Check if requested quantity exceeds available batches
        $totalBatchQuantity = ProductChange::where('product_id', $item->id)->sum('quantity');
       /* if ($item->quantity > $totalBatchQuantity) {
            return response()->json([
                'status' => 0,
                'message' => 'Requested quantity for ' . $product->name . ' exceeds available batches. Available: ' . $totalBatchQuantity
            ]);
        }
        */
    }
    

    // Only proceed if all items have sufficient stock
    $data = [
        'total_amount'     => $request->input('total_amount'),	
        'discounted_total' => $request->input('discounted_total'),	
        'create_user'      => $request->input('waiter_id') ?? auth()->id(),	
        'comment'          => $request->input('comment'),	
        'amount_paid'      => $request->input('amount_paid') ?? 0,
        'amount_remain'    => $request->input('amount_remain'),	
        'discount_perc'    => $request->input('discount'),
        'discount_amount'  => $request->input('discount_amount'),
        'payment_date'     => $request->input('payment_date'),
        'client_id'        => is_numeric($request->input('client_id')) ? $request->input('client_id') : NULL,	
        'paid'             => $request->input('amount_remain') == 0,
        'committed_date'   => $request->input('committed_date') ?? date('Y-m-d')
    ];

    if ($request->has('id')) {
        $order = Sale::find($request->input('id'));
        $this->draftSaleItems($order, 'edit');
        Sale::where('id', $order->id)->update($data);
    } else {
        $data = array_merge($data, [
            'type'        => 'POS_SALE',
            'reference'   => generateRowCode(8),
            'code' => createTransactCode('CI')
        ]);
        $order = Sale::create($data);
    }
    
    if ($request->input('amount_paid') != 0) {
        $amountPaid = $request->input('amount_paid') ?? 0;
        Payment::create([
            'committed_date' => $request->input('committed_date'),	
            'transaction_id' => $order->id,	
            'payment_type'   => $request->input('payment_method'),
            'amount_paid'    => $amountPaid,	
            'comment'        => $request->input('comment'),
            'reference'      => $request->input('payment_ref'),
            'create_user'    => auth()->id()
        ]);
    }

    foreach($items as $item) {
        SaleItem::create([
            'sale_id'  => $order->id,
            'item_id'  => $item->id,
            'batch'    => $item->batch ?? null,
            'quantity' => $item->quantity,
            'price'    => $item->price,
            'amount'   => $item->quantity * $item->price,
            'comment'  => $item->comment ?? NULL
        ]);
        
        $product = Product::find($item->id);
        // Only deduct stock for storable products (services have no inventory)
        if ($product && $product->is_storable) {
            $product->quantity -= $item->quantity;
            $product->save();
            
            $totalQuantity = 0;
            foreach ($product->ProductChanges as $productChange) {
                $totalQuantity += $productChange->quantity;
            }

            if ($item->quantity <= $totalQuantity) {
                $productChange = ProductChange::where('product_id', $item->id)
                    ->where('batch_nber', $item->batch)
                    ->first();
        
                if (!$productChange) {
                    return response()->json([
                        'status'=>0,
                        'message'=> 'Batch are you trying search is unvailable!'
                    ]);
                }
            
                $remainingQuantity = $item->quantity;
            
                if ($productChange->quantity >= $remainingQuantity) {
                    $productChange->decrement('quantity', $remainingQuantity);
                    $remainingQuantity = 0;
                } else {
                    $remainingQuantity -= $productChange->quantity;
                    $productChange->update(['quantity' => 0]);
                }

                foreach($product->ProductChanges as $otherProductChange){
                    if ($remainingQuantity <= 0) break;
                    if ($otherProductChange->quantity >= $remainingQuantity) {
                        $otherProductChange->decrement('quantity', $remainingQuantity);
                        $remainingQuantity = 0;
                    } else {
                        $remainingQuantity -= $otherProductChange->quantity;
                        $otherProductChange->update(['quantity' => 0]);
                    }
                }
            }
        }
    }
    
    return response()->json([
        'status'  => 1,
        'message' => 'Save sucessfully',
        'order'   => Sale::selectRaw('id, reference, committed_date, discounted_total, amount_paid, amount_remain, client_id, paid, branch_id, create_user, comment, payment_date')
                             ->where('sales.id', $order->id)->with('client', 'creator', 'items')->first()
    ]);
}

    /**
     * Handle Delete
     * @return JsonResponse
     */
    public function handleDestroy($saleId)
    {
        $sale = Sale::find($saleId);
        if (!$sale) {
            return response()->json([
                'status' => 0,
                'error'  => 'No record found'
            ], 404);
        }
        $this->draftSaleItems($sale);
        $sale->delete();
        return response()->json([
            'status' => 1,
            'message' => 'Deleted successfully'
        ]);
    }

    /**
     * Delete made payments for a particular sale
     * Delete sale Item
     * @param Modules\Api\Models\Sale $saleId
     * @return void
     */
    private function draftSaleItems(Sale $sale, $action = 'delete')
    {
        $items = SaleItem::where('sale_id', $sale->id)->get();
        foreach ($items as $row) {
            $product = Product::find($row->item_id);
            if ($product && $product->is_storable) {
                $product->quantity += $row->quantity;
                $product->save();
            }
            if ($action == 'edit') {
                DB::table('sale_items')->where('id', $row->id)->delete();
            } else {
                $row->delete();
            }
        }
        /** Delete all payments
         *     $payments = Payment::where('transaction_id', $sale->id)->get();
   
         */
    
        Payment::where('transaction_id', $sale->id)->forceDelete();
        /*foreach ($payments as $payment) {
            if ($action == 'edit') {
                //$payment->forceDelete();
                DB::table('payments')->where('id', $payment->id)->delete();
            } else {
                $payment->delete();
            }
        }
        */
    }

    /**
     * Get Sales Items
     * @param int $id
     * @return ResponseJson
     */
    public function getSaleItems($id)
    {
        $items = SaleItem::selectRaw('sale_items.item_id','sale_items.batch','sale_items.quantity, sale_items.price, sale_items.amount as total_amount, products.name,products.packaging, products.code, products.pcode, units.name As unit, stockin_histories.expiration_date')
                            ->join('products', 'sale_items.item_id', '=', 'products.id')
                            ->leftJoin('units', 'products.unit_id', '=', 'units.id')
                            ->leftJoin('stockin_histories', 'stockin_histories.product_id', '=', 'products.id')
                            ->where('sale_id', $id)
                            ->get();
        //$items = $items->makeHidden(['product']);
        return response()->json([
            'status'  => 1,
            'items'   => $items
        ]);
    }
  
    /** 
     * Get sale Collection
     * @param $reference
     * @return @return \Illuminate\Support\Collection
     */

     public function getSaleData(string $reference)
     {
        $sale = Sale::selectRaw('id, reference, code, committed_date, payment_date, discounted_total, amount_paid, amount_remain, client_id, paid, branch_id, create_user')
                    ->where('reference', $reference)
                    ->with('client', 'creator')
                    ->first();

        if (!$sale) {
            return response()->json([
                'status' => 0,
                'message' => 'Sale not found',
                'record' => null,
                'items' => []
            ], 404);
        }

        $sale->makeVisible(['payments']);

        $saleItems = SaleItem::selectRaw('sale_items.item_id, sale_items.batch, sale_items.quantity, sale_items.price, sale_items.amount as total_amount, products.name, products.packaging, products.code, products.pcode, units.name As unit, products.id')
            ->join('products', 'sale_items.item_id', '=', 'products.id')
            ->leftJoin('units', 'products.unit_id', '=', 'units.id')
            ->where('sale_id', $sale->id)
            ->get();

        $items = $saleItems->map(function ($row) {
            $product = Product::with('ProductChanges')->find($row->item_id);
            $productChanges = $product ? $product->ProductChanges->toArray() : [];
            $isStorable = $product ? (int) $product->is_storable : 1;
            return [
                'id'              => $row->item_id,
                'item_id'         => $row->item_id,
                'name'            => $row->name,
                'packaging'       => $row->packaging,
                'batch'           => $row->batch,
                'quantity'        => (int) $row->quantity,
                'qty'             => $isStorable && $product ? ($product->quantity ?? 0) : 9999,
                'price'           => (float) $row->price,
                'total_amount'    => (float) $row->total_amount,
                'code'            => $row->code,
                'pcode'           => $row->pcode,
                'unit'            => $row->unit,
                'product_changes' => $productChanges,
                'is_storable'     => $isStorable,
            ];
        });

        return response()->json([
            'status' => 1,
            'record' => $sale,
            'items'  => $items
        ]);
     }

     /**
      * Get Payment methods
      * @return Jsonresponse
      */
      public function getPaymentMethod()
      {
        return response()->json([
            'status' => 1,
            'rows'   => PaymentMethod::get()
        ]);
      }

      
     /**
      * Get Payment methods
      * @return Jsonresponse
      */

      public function getPaymentHistory(Request $request)
      {
        $result = Payment::select('*');

        $from = date('Y-m-d');
     /*
        if(!empty($fromd = $request->get('from'))){
            $from = $fromd;
        }
*/
        $to = $request->get('to');
        if (empty($to)){
            $result->where('created_at', 'like', '%' . $from . '%');
        } else {
            $result->where('created_at', '>=', $from)
                    ->where('created_at', '<=', $to);
        }

        if (!empty($payment_type = $request->get('payment_type'))) {
            $result->where('payment_type', $payment_type);
        }
        
        return response()->json([
            'status' => 1,
            'rows'   => $result->with('transaction', 'creator')
                               ->orderBy('id', 'DESC')
                               ->paginate(45),
        ]);
      }

      /**
       * Handle Partial Payment
       */
    public function handlePartialPayment(Request $request)
    {
        $row = Sale::findOrFail($request->input('record_id'));
        if(!empty($row)){
            $row->amount_paid += $request->input('amount_paid');
            $row->amount_remain -= $request->input('amount_paid');
            $row->paid = $request->input('amount_remain') <= 0;
            $row->save();
            Payment::create([
                'committed_date' => $request->input('committed_date') ?? $request->input('payment_date'),	
                'transaction_id' => $row->id,	
                'payment_type'   => $request->input('payment_method'),
                'amount_paid'    => $request->input('amount_paid') ?? 0,	
                'comment'        => $request->input('comment'),
                'reference'      => $request->input('payment_ref'),
                'create_user'    => auth()->id()
            ]);
        }
        return response()->json([
            'status'  => 1,
            'message' => 'Record saved successfully'
        ]);
    }
}